----------------
title : C++ 레퍼런스 - string 의 append 함수
cat_title : append
ref_title : append
path : /C++ Reference/string
publish_date : 2020-03-10
----------------

##@ cpp-ref-start

#@ append

```cpp-formatted
// C++ 20 부터 아래 모든 함수들은 constexpr
basic_string& append(size_type count, CharT ch);  // (1)
basic_string& append(const basic_string& str);    // (2)
basic_string& append(const basic_string& str, size_type pos,
                     size_type count = npos);           // (3)
basic_string& append(const CharT* s, size_type count);  // (4)
basic_string& append(const CharT* s);                   // (5)
template <class InputIt>
basic_string& append(InputIt first, InputIt last);         // (6)
basic_string& append(std::initializer_list<CharT> ilist);  // (7)

// 아래 둘은 C++ 17 에 추가됨
template <class T>
basic_string& append(const T& t);  // (8)
template <class T>
basic_string& append(const T& t, size_type pos, size_type count = npos);  // (9)
```

문자열 끝에 문자들을 덧붙인다.

1. `count` 개의 문자 `ch` 를 뒤에 추가한다.
2. 문자열 `str` 을 뒤에 추가한다.
3. `str` 의 부분 문자열 (`pos` 부터 `count` 개 만큼) 을 문자열 뒤에 붙인다. 쉽게 생각해서 `str.substr(pos, count)` 한 것을 추가한다고 보면 된다.
4. `s` 부터 `s + count` 전 까지의 문자들을 문자열 뒤에 붙인다. 이 때 해당 구간안에 `NULL` 문자가 있어도 괜찮다.
5. 널 종료 문자열 `s` 를 문자열 뒤에 붙인다.
6. `first` 부터 `last` 전 까지의 문자들을 문자열 뒤에 붙인다.
7. 초기화자 리스트 (`initializer_list`) 안에 문자들을 뒤에 붙인다.
8. `t` 를 `string_view` 로 캐스팅 한 뒤에 해당 `string_view` 안에 있는 문자들을 문자열 뒤에 붙인다. 이 때, `T` 는 `string_view` 로 타입 변환이 가능해야 한다. 참고로 이 함수는 `T` 가 `const CharT*` 로 변환가능하지 않을 때 선택된다 (만일 변환 가능할 경우 (5) 번 오버로딩이 선택됨). 다시 말해 `std::is_convertible_v<const T&, std::basic_string_view<CharT, Traits>>` 가 참이고 `std::is_convertible_v<const T&, const CharT*>` 는 거짓 이어야 한다.
9. (8) 과 동일하지만 `pos` 부터 `count` 개 만큼의 문자들을 복사한다.

### 인자들

* `count` : 추가할 문자들의 개수
* `pos` : 첫 번째 문자가 삽입될 위치.
* `ch` : 추가할 문자
* `first, last` : 추가할 문자들의 범위. `first` 부터 `last` **전** 까지의 문자들이 들어간다.
* `str` : 추가할 문자열
* `s` : 추가할 문자들을 가리키는 포인터
* `ilist` : 추가할 문자들을 나타내는 초기화자 리스트.
* `t` : `std::basic_string_view` 로 변환 가능한 객체로, 해당 객체는 `string_view` 로 변환 된 후 문자열 뒤에 추가된다.

### 리턴값

`*this`

### 예외

만일 예외가 `throw` 된다면 이 함수는 아무 작업도 수행하지 않는다.

`size() > max_size()` 일 경우 `std::length_error` 을 던진다.

### 실행 예제

```cpp-formatted
#include <iostream>
#include <string>

int main() {
  std::basic_string<char> str = "string";
  const char* cptr = "C-string";
  const char carr[] = "Two and one";

  std::string output;

  // 1) 문자 '*' 를 3 번 추가한다.
  output.append(3, '*');
  std::cout << "1) " << output << "\n";

  //  2) 전체 문자열 (str)을 추가한다.
  output.append(str);
  std::cout << "2) " << output << "\n";

  // 3) 문자열 str 의 일부분 (이 경우 마지막 3 문자 "ing") 을 추가한다.
  output.append(str, 3, 3);
  std::cout << "3) " << output << "\n";

  // 4) 널 종료 문자열의 일부분을 추가한다.
  // 참고로 append 가 *this 를 리턴하기 때문에 append 를 여러번 이어서 호출할 수
  // 있다.
  output.append(1, ' ').append(carr, 4);
  std::cout << "4) " << output << "\n";

  // 5) 널 종료 문자열을 추가한다.
  output.append(cptr);
  std::cout << "5) " << output << "\n";

  // 6) 범위 안의 문자열을 추가한다. (6) 번 오버로딩이 호출됨.
  output.append(&carr[3], std::end(carr));
  std::cout << "6) " << output << "\n";

  // 7) 초기화자 리스트 안의 문자열을 추가한다.
  output.append({' ', 'l', 'i', 's', 't'});
  std::cout << "7) " << output << "\n";
}
```

```exec
1) ***
2) ***string
3) ***stringing
4) ***stringing Two 
5) ***stringing Two C-string
6) ***stringing Two C-string and one
7) ***stringing Two C-string and one list
```

### 참고 자료

* `operator+=` : 문자열 뒤에 문자들을 추가한다.
* `strcat` : 두 문자열을 하나로 합친다.
* `strncat` : 두 문자열의 일부분을 합친다.